/*******************************************************************************
 *
 *  hal_usb.c
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/

#include "msp430.h"
#include "hal_usb.h"

#if defined(__TI_COMPILER_VERSION__)
unsigned char g_ucRxBuffer[UART_IF_BUFFER];
#else
__no_init unsigned char g_ucRxBuffer[UART_IF_BUFFER];
#endif

volatile unsigned short g_usRxBufferCounter = 0;
volatile unsigned char uart_have_cmd =0;

/**********************************************************************//**
* @brief  Initializes the serial communications peripheral and GPIO ports
*         to communicate with the TUSB3410.
*
* @param  none
*
* @return none
**************************************************************************/

void halUsbInit(void)
{
    unsigned short i;

    for (i = 0; i < UART_IF_BUFFER; i++)
        g_ucRxBuffer[i] = '\0';

	g_usRxBufferCounter = 0;
	uart_have_cmd = 0;
	
    UART_PORT_SEL |= UART_PIN_RXD + UART_PIN_TXD;
    UART_PORT_DIR |= UART_PIN_TXD;
    UART_PORT_DIR &= ~UART_PIN_RXD;

    UCA1CTL1 |= UCSWRST;                    //Reset State
    UCA1CTL0 = UCMODE_0;

    UCA1CTL0 &= ~UC7BIT;                    // 8bit char
    UCA1CTL1 |= UCSSEL_2;
    UCA1BR0 = 0x82;                           // 16Mhz/115200=139 = 0x8B
    UCA1BR1 = 0x06;							// 16Mhz/9600=1666.67 =0x682
    UCA1MCTL = 0xE;
    UCA1CTL1 &= ~UCSWRST;
    UCA1IE |= UCRXIE;

    __bis_SR_register(GIE);                 // Enable Interrupts

}

/**********************************************************************//**
* @brief  Disables the serial communications peripheral and clears the GPIO
*         settings used to communicate with the TUSB3410.
*
* @param  none
*
* @return none
**************************************************************************/

void halUsbShutDown(void)
{
    UCA1IE &= ~UCRXIE;
    UCA1CTL1 = UCSWRST;                     //Reset State
    UART_PORT_SEL &= ~(UART_PIN_RXD + UART_PIN_TXD);
    UART_PORT_DIR |= UART_PIN_TXD;
    UART_PORT_DIR |= UART_PIN_RXD;
    UART_PORT_OUT &= ~(UART_PIN_TXD + UART_PIN_RXD);
}

/**********************************************************************//**
* @brief  Sends a character over UART to the TUSB3410.
*
* @param  character The character to be sent.
*
* @return none
**************************************************************************/

void halUsbSendChar(char character)
{
    while (!(UCA1IFG & UCTXIFG)) ;
    UCA1TXBUF = character;
}

/**********************************************************************//**
* @brief  Sends a string of characters to the TUSB3410
*
* @param  string[] The array of characters to be transmit to the TUSB3410.
*
* @param  length   The length of the string.
*
* @return none
**************************************************************************/

void halUsbSendString(char string[], unsigned char length)
{
    unsigned char i;

    for (i = 0; i < length; i++)
        halUsbSendChar(string[i]);
}

/************************************************************************/
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR(void)
{
    g_ucRxBuffer[g_usRxBufferCounter] = UCA1RXBUF;
	//if(g_ucRxBuffer[g_usRxBufferCounte] == 0x0D )
		uart_have_cmd = g_ucRxBuffer[1];
	if(++g_usRxBufferCounter >= UART_IF_BUFFER)
		g_usRxBufferCounter = 0;
	

}

